<?php

if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Web_Payment_Gateway class.
 *
 * @since 2.0.0
 * @extends WC_Payment_Gateway
 */
class WC_Web_Payment_Gateway extends WC_Payment_Gateway
{

    protected $apimethod = array(
        'checkout' => '/api/v1/session',
        'refund'   => '/api/v1/payment/refund'
    );

    protected $lastHttpStatus;
    
    /**
     * @var WC_Order_Refund
     */
    protected $refundObject;

    public $url;
    public $method;
    public $secret;
    public $password;
    public $display_icon;
    public $vat_calculation;
    public $iframe;
    public $form_id;
    public $url_target;
    public $preferred_iframe_hook;
    public $order_status_after_payment;
    public $additional_payment_parameters;
    public $additional_payment_parameters_kunnect;

    public $debug = 'no';

    public $currencies_3dotexponent = ['BHD', 'JOD', 'KWD', 'OMR', 'TND'];
    public $currencies_noexponent = [
        //'CLP', 
        'VND',
        'ISK',
        'UGX',
        //'KRW', 
        //'JPY'
    ];

    public static $cryptoCurencies = [
        'BTC' => 'BTC',
        'ETH' => 'ETH',
        'USDT' => 'USDT ERC-20',
        'USDC' => 'USDC ERC-20',
        'USDT_POL' => 'USDT Polygon',
        'USDC_POL' => 'USDC Polygon',
        'USDT_TRX' => 'USDT TRC-20',
    ];

    public function __construct()
    {
  
        // Register plugin information
        $this->id = 'wpgfull';
        $this->has_fields = true;
        $this->supports = array(
            'refunds',
            'products',
            'tokenization',
        );

        // Create plugin fields and settings
        $this->init_form_fields();
        $this->init_settings();

        // Get setting values
        foreach ($this->settings as $key => $val) {
            $this->$key = $val;
        }


        if (empty($this->icon)) {
            // Load plugin checkout default icon
            $this->icon = WC_WPG_PLUGIN_URL . '/images/default-logo.png';
        }

        if (version_compare(WC_VERSION, '9.8.5', '>=') && version_compare(WC_VERSION, '10.0.0', '<')) {
            $this->method_title = $this->title;
            $this->method_description = $this->description;
        }


        //image select
        add_action('admin_enqueue_scripts', array($this, 'loadscripts'));


        // Add hooks
        add_action($this->preferred_iframe_hook, array($this, 'receipt_page'), 10, 1);

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

        add_action('admin_notices', array($this, 'wpgfull_commerce_ssl_check'));

        //Save backward compatibility
        add_action('woocommerce_api_wc_web_payment_gateway', array($this, 'check_ipn_response_backward'));
        add_action('admin_notices', [$this, 'show_notice_change_webhook']);

        add_filter('woocommerce_shared_settings', [$this, 'modifyCheckoutFieldsNewHook'], 30);
        add_filter('the_content',  [$this, 'editCheckoutContent']);
       
        add_action('woocommerce_checkout_create_order', [$this, 'saveKunnectCryptoCurrency'], 20, 2);

        add_action('woocommerce_create_refund', [$this, 'setRefundInfo'], 10, 2);
    }


    public function get_new_image()
    {
        if (isset($_GET['id'])) {
            $data = array(
                'image'    => wp_get_attachment_image(filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT), array(90, 20), false, array('id' => 'wpg_custom_logo_prewiev')),
                'imageurl' => wp_get_attachment_image_url(filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT), array(90, 20), true),
            );
            wp_send_json_success($data);
        } else {
            wp_send_json_error();
        }
    }

    function loadscripts($page)
    {
        if ($page == 'woocommerce_page_wc-settings') {
            wp_enqueue_media();
            wp_enqueue_script('wpg_load_logo', WC_WPG_PLUGIN_DIR . 'js/wpg_load_logo.js', array('jquery'));
        }
    }


    /**
     * Refund function for gateway if suppors
     **/

    public function process_refund($order_id, $amount = null, $reason = '')
    {
        $order = wc_get_order($order_id);
        if (!is_a($order, \WC_Order::class)) {
            return false;
        }

        $transaction_id = $order->get_transaction_id();

        if (empty($transaction_id)) {
            return new WP_Error('order_not_payed',  __('Payment ID missing', 'woocommerce-gateway-wpgfull'));
        }

        $refundsInfo = $order->get_meta('wpg_refunds', true);
        if (!is_array($refundsInfo)) {
            $refundsInfo = [];
        }


        foreach ($refundsInfo as $refundInfo) {
            if ($refundInfo['new'] == true) {
                 return new WP_Error('have_open_refund_request',  __('You have an open refund request. Please wait for it to be completed.', 'woocommerce-gateway-wpgfull'));
            }
        }

        //The refund amount is validated by WooCommerce in wc_create_refund function.

        $correct_amount = number_format($amount, 2, '.', '');
        if (in_array(get_woocommerce_currency(), $this->currencies_noexponent)) {
            $correct_amount = number_format($order->get_total(), 0, '.', '');
        } elseif (in_array(get_woocommerce_currency(), $this->currencies_3dotexponent)) {
            $correct_amount = number_format($order->get_total(), 3, '.', '');
        }

        $hash = sha1(md5(strtoupper($transaction_id . $correct_amount . $this->password)));
        
        $requestBody = array(
            'merchant_key' => $this->secret,
            'payment_id'    => $transaction_id,
            'amount'      => $correct_amount,
            'hash'         => $hash
        );

        $action_adr = rtrim($this->url, '/') . $this->apimethod['refund'];

        $response = $this->makeRequest($action_adr, $requestBody);

        if (isset($response['error_message'])) {
            return new WP_Error('gateway_error',  sprintf(__('Gateway error: %s', 'woocommerce-gateway-wpgfull'), $response['error_message']));
        }

        if ($response['result'] == 'accepted' && $response['payment_id'] == $transaction_id) {
          

            $refundsInfo[] = ['wc_refund_id' => $this->refundObject->get_id(), 'payment_id' => $response['payment_id'], 'amount' => $correct_amount, 'new' => true];

            $order->add_meta_data('wpg_refunds', $refundsInfo, true);
            $order->add_order_note(sprintf(__('Refund request created. Refund ID: %s. If the request is unsuccessful, the refund will be canceled.', 'woocommerce-web-payment-gateway'), $response['payment_id']));
            $order->save();

            return true;
        }

        return false;
    }

    public function check_ipn_response_backward()
    {
        $this->log('Legacy POST api: ' . print_r($_POST, true));
        if ($_SERVER['REQUEST_METHOD'] != 'POST') {
            exit;
        }

        $params = $_POST;
        $hash_string = $params['id'] . $params['order_number'] . $params['order_amount'] . $params['order_currency'] . $params['order_description'] . $this->password;
        $hash = sha1(md5(strtoupper($hash_string)));
        if ($params['hash'] != $hash) {
            exit;
        }

        $this->update_option('show_legacy_notice', '1');

        $order_id = $params['order_number'];
        $order = wc_get_order($order_id);

        if ($order == false) {
            $this->log('Recieved callback for deleted order: ' . $order_id);
            return;
        }

        if ($params['type'] == 'sale') {
            $this->processSimpleOrder($params, $order);
        } elseif ($params['type'] == 'refund') {
            $this->processRefundCallback($params, $order);
        }

    }

    public function show_notice_change_webhook()
    {
        $opt = $this->get_option('show_legacy_notice', '0');
        if ('yes' == $this->enabled && $opt == '1') {
            $admin_url = admin_url('admin.php?page=wc-settings&tab=checkout&section=wpgfull');
            echo '<div class="notice notice-error is-dismissible"><p>' . sprintf(__('WPG: Update the callback url in your merchant settings. <a href="%s">You can find the new link here</a>', 'woocommerce-gateway-wpgfull'), $admin_url) . '</p></div>';
        }
    }

    public function check_ipn_response(WP_REST_Request $request)
    {
        $this->log('POST api: ' . print_r($request->get_params(), true));


        if ($request->get_method() != 'POST') {
            exit;
        }

        $params = $request->get_params();

        $hash_string = $params['id'] . $params['order_number'] . $params['order_amount'] . $params['order_currency'] . $params['order_description'] . $this->password;
        $hash = sha1(md5(strtoupper($hash_string)));

        if ($params['hash'] != $hash) {
            return ['error' => 'hash mismatch'];
        }

        $this->update_option('show_legacy_notice', '0');

        $order_id = $params['order_number'];
        $order = wc_get_order($order_id);

        if ($order == false) {
            $this->log('Recieved callback for deleted order: ' . $order_id);
            return;
        }

        if ($params['type'] == 'sale') {
            $this->processSimpleOrder($params, $order);
        } elseif ($params['type'] == 'refund') {
            $this->processRefundCallback($params, $order);
        }
    }

    protected function processSimpleOrder($params, $order)
    {
        $order->set_transaction_id($params['id']);

        if (isset($params['card_token']) && $order->get_user_id()) {

            $savedTokens = WC_Payment_Tokens::get_customer_tokens($order->get_user_id());
            foreach ($savedTokens as $savedToken) {
                if ($savedToken->get_token() == $params['card_token']) {
                    $cardtoken = $savedToken;
                    break;
                }
            }

            if (!isset($cardtoken)) {
                $cardtoken = new WC_Payment_Token_CC();
                $cardtoken->set_token($params['card_token']);
                $cardtoken->set_last4(substr($params['card'], -4));
                
                $expiration = explode('/', $params['card_expiration_date']);
                
                $cardtoken->set_expiry_month($expiration[0]);
                $cardtoken->set_expiry_year($expiration[1]);
                $cardtoken->set_card_type($this->getCardTypeFromBin($params['card']));
                $cardtoken->set_gateway_id($this->id);
                $cardtoken->set_user_id($order->get_user_id());

                $cardtoken->save();

                $order->add_payment_token($cardtoken);
                $order->save();
            }
        }

        if ($order->get_status() == 'pending' || $order->get_status() == 'waiting' || $order->get_status() == 'failed' || $order->get_status() == '') {

            $order->set_transaction_id($params['id']);

            if ($params['status'] == 'success' && $params['type'] == 'sale') {
                //successful purchase
                //$woocommerce->cart->empty_cart();
                $order->update_status($this->order_status_after_payment, 'Payment successfully paid'); //completed or processing
                //$order->payment_complete($_POST['id']);

                if (isset($params['vat_amount'])) {
                    $order->add_meta_data('wpg_vat_amount', $params['vat_amount']);
                    $order->save();
                }

                exit;
            }

            if ($params['status'] == 'waiting' && $params['type'] == 'sale') {
                //waiting purchase
                $order->update_status('on-hold', __('On hold', 'woocommerce'));
                exit;
            }

            if ($params['status'] == 'fail' && $params['type'] == 'sale') {
                //failed purchase
                $order->update_status('failed', $params['reason']);
                exit;
            }
        }
    }

    protected function processRefundCallback($params, $order)
    {
        if (in_array($order->get_status(), ['completed', 'processing', 'refunded'])) {

            $refundsInfo = $order->get_meta('wpg_refunds', true);

            foreach ($refundsInfo as $key => $refundInfo) {
                if ($refundInfo['amount'] == $params['order_amount'] && $refundInfo['new'] == true) {
                    $findedRefundsKey = $key;
                    break;
                }
            }

            if (isset($findedRefundsKey)) {
                $wcRefund = new WC_Order_Refund($refundsInfo[$findedRefundsKey]['wc_refund_id']);

                if ($params['status'] == 'success') {
                    //$order->update_status('refunded', __('Refunded', 'woocommerce'));
                    $order->add_order_note(__('Refund confirmed by the payment system.', 'woocommerce-gateway-wpgfull'));
                    $refundsInfo[$findedRefundsKey]['new'] = false;
                }

                if ($params['status'] == 'fail') {
                    $order->add_order_note(sprintf(__('Refund request failed. Refund removed. Reason: %s', 'woocommerce-gateway-wpgfull'), $params['reason']));
                    $wcRefund->delete();
                    unset($refundsInfo[$findedRefundsKey]);
                }
                if ($params['status'] == 'waiting') {
                    $order->add_order_note(__('Refund request has status Waiting.', 'woocommerce-gateway-wpgfull'));
                }
                if ($params['status'] == 'undefined') {
                    $order->add_order_note(__('Refund request undefined. Refund removed', 'woocommerce-gateway-wpgfull'));
                    $wcRefund->delete();
                    unset($refundsInfo[$findedRefundsKey]);
                }
            }

            $order->add_meta_data('wpg_refunds', $refundsInfo, true);
            $order->save();
        }
    }

    public function setRefundInfo($refund, $args)
    {
        $this->refundObject = $refund;
    }


    /**
     * Check if SSL is enabled and notify the user.
     */
    function wpgfull_commerce_ssl_check()
    {
        //if (!$_POST)
        //if ('no' == get_option('woocommerce_force_ssl_checkout') && 'yes' == $this->enabled) {
        if (is_ssl() == false && 'yes' == $this->enabled) {
            $admin_url = admin_url('admin.php?page=wc-settings&tab=checkout');
            echo '<div class="notice notice-error is-dismissible"><p>' . sprintf(__('WPG is enabled. But <a href="%s">force SSL option</a> is disabled. Your checkout is not secure! Please enable SSL and ensure your server has a valid SSL certificate.', 'woocommerce-gateway-wpgfull'), $admin_url) . '</p></div>';
        }
    }

    /**
     * Initialize Gateway Settings Form Fields.
     */
    function init_form_fields()
    {
        $orderStatuses = wc_get_order_statuses();

        unset($orderStatuses['wc-pending']);
        unset($orderStatuses['wc-on-hold']);
        unset($orderStatuses['wc-cancelled']);
        unset($orderStatuses['wc-refunded']);
        unset($orderStatuses['wc-failed']);
        unset($orderStatuses['wc-checkout-draft']);

        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'woocommerce-gateway-wpgfull'),
                'label' => __('Enable WPG Commerce', 'woocommerce-gateway-wpgfull'),
                'type' => 'checkbox',
                'description' => '',
                'default' => 'no',
            ),
            'title' => array(
                'title' => __('Title', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'woocommerce-gateway-wpgfull'),
                'default' => __('WPG Commerce', 'woocommerce-gateway-wpgfull'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Front description', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'default' => __('Pay online through WPG Commerce', 'woocommerce-gateway-wpgfull'),
            ),
            'method_description' => array(
                'title' => __('Description', 'woocommerce-gateway-wpgfull'),
                'type' => 'textarea',
                'description' => __('WPG redirects customers enter payment details.', 'woocommerce-gateway-wpgfull'),
                'default' => __('You can make a payment through the Web Payment Gateway system', 'woocommerce'),
            ),
            'url' => array(
                'title' => __('Checkout host', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'description' => __('Url from payment system to send a payment request', 'woocommerce-gateway-wpgfull'),
            ),
            'method' => array(
                'title' => __('Payment method', 'woocommerce-gateway-wpgfull'),
                'type' => 'multiselect',
                'description' => __('Payment method that client uses', 'woocommerce-gateway-wpgfull'), /////
                'options' => array(
                    'card' => 'Card',
                    'googlepay' => 'Google Pay',
                    'paypal' => 'Pay Pal',
                ),
            ),
            'additional_payment_parameters' => array(
                'title' => __('Additional payment parameters', 'woocommerce-gateway-wpgfull'),
                'type' => 'multiselect',
                'description' => __('Include additional payment parameters to api request', 'woocommerce-gateway-wpgfull'), /////
                'options' => array(
                    'card_order_info' => 'Card(Order info)',
                    'kunnect' => 'Cryptocurrency (mandatory)'
                ),
            ),
             'additional_payment_parameters_kunnect' => array(
                'title' => __('Choose cryptocurrencies', 'woocommerce-gateway-wpgfull'),
                'type' => 'multiselect',
                'description' => __('List of allowed cryptocurrencies for user selection', 'woocommerce-gateway-wpgfull'), /////
                'options' => self::$cryptoCurencies,
            ),
            'secret' => array(
                'title' => __('Merchant key', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'description' => __('Merchant key from payment system for customer identification', 'woocommerce-gateway-wpgfull'),
            ),
            'password' => array(
                'title' => __('Merchant password', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'description' => __('Merchant password from payment system.', 'woocommerce-gateway-wpgfull'),
            ),

            'vat_calculation' => array(
                'title' => __('VAT Calculation', 'woocommerce-gateway-wpgfull'),
                'label' => __('Enable/Disable', 'woocommerce-gateway-wpgfull'),
                'type' => 'checkbox',
                'description' => __('The tax is calculated by the payment system. The order amount is passed tax-free, regardless of Woocommerce Tax settings.', 'woocommerce-gateway-wpgfull'),
                'default' => 'no',
            ),

            'iframe' => array(
                'title' => __('Iframe payment', 'woocommerce-gateway-wpgfull'),
                'label' => __('Enable/Disable', 'woocommerce-gateway-wpgfull'),
                'type' => 'checkbox',
                'description' => __('Show payment form in iframe after checkout.', 'woocommerce-gateway-wpgfull'),
                'default' => 'no',
            ),

            'url_target' => array(
                'title' => __('Url target', 'woocommerce-gateway-wpgfull'),
                'type' => 'select',
                'description' => __('A browsing context where Customer should be returned', 'woocommerce-gateway-wpgfull'),
                'options' => array(
                    'noused' => 'Not used',
                    'self' => 'Self',
                    'blank' => 'Blank',
                    'parent' => 'Parent',
                    'top' => 'Top',
                ),
                'default' => 'noused',
            ),

            'preferred_iframe_hook' => array(
                'title' => __('Display hook', 'woocommerce-gateway-wpgfull'),
                'type' => 'select',
                'description' => __('Selection of the hook on which to show the iframe'),
                'options' => array(
                    'woocommerce_thankyou_wpgfull' => 'woocommerce_thankyou',
                    'woocommerce_order_details_after_customer_details' => 'woocommerce_order_details_after_customer_details',
                ),
                'default' => 'woocommerce_thankyou_wpgfull',
            ),

            'form_id' => array(
                'title' => __('Form id', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'description' => __('Id form to show your own design, approved by the manager.', 'woocommerce-gateway-wpgfull'),
            ),

            'order_status_after_payment' => array(
                'title' => __('Order status', 'woocommerce-gateway-wpgfull'),
                'type' => 'select',
                'description' => __('Select the order status that will be set after successful payment', 'woocommerce-gateway-wpgfull'),
                'options' => $orderStatuses,
                'default' => 'wc-completed',
            ),

            'debug' => array(
                'title' => __('Enable debug', 'woocommerce-gateway-wpgfull'),
                'type' => 'text',
                'default' => 'no',
                'description' => __('Enable debug logs', 'woocommerce-gateway-wpgfull'),
            ),
            'display_icon' => array(
                'title' => __('Hide icon on front', 'woocommerce-gateway-wpgfull'),
                'type' => 'checkbox'
            ),
            'icon' => array(
                'type' => 'hidden',
                'class' => 'wpg_custom_logo',
            ),
        );
    }

    /**
     * UI - Admin Panel Options
     */
    function admin_options()
    {

?>
        <h3><?php _e('Web Payment Gateway Settings', 'woocommerce-gateway-wpgfull'); ?></h3>
        <p><?php _e('Web Payment Gateway. The plugin works by opened checkout page, and then sending the details to payment system for verification.', 'woocommerce-gateway-wpgfull'); ?></p>
        <input type="hidden" name="" id="wc_api_url" value="<?php echo rest_url(trailingslashit( 'wpgfull/v1' ) . 'setpicture'); ?>">
        <table class="form-table">
            <?php
            $this->generate_settings_html();
            ?>
            <p>
                <strong><?php _e('Callback Url: ') ?></strong><?php echo rest_url(trailingslashit( 'wpgfull/v1' ) . 'callback'); ?>
            </p>
            <!--  -->
            <tr valign="top" class="">
                <th scope="row" class="titledesc">
                    <label for="myprefix_media_manager">Logo </label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span>Preview</span></legend>
                        <img id="wpg_custom_logo_prewiev" width="90" height="20" src="<?= $this->icon ?>" alt="<?= $this->title ?>">
                        <input type='button' class="button-primary" value="<?php esc_attr_e('Select a image', 'mytextdomain'); ?>" id="myprefix_media_manager" />
                        <p class="description">Custom logo payment system.</p>
                    </fieldset>
                </td>
            </tr>
            <!--  -->
        </table>


    <?php
    }


    /**
     * Process the payment and return the result.
     *
     * @param int @order_id
     * @return array
     */
    public function process_payment($order_id)
    {

        $order = new WC_Order($order_id);
        $this->allowTokenize();

        if ($this->iframe == 'yes') {
            return array(
                'result' => 'success',
                'redirect' => $this->get_return_url($order),
            );
        } else {
            return array(
                'result' => 'success',
                'redirect' => $this->generate_form($order)
            );
        }
    }


    /**
     * Display information on the Thank You page
     *
     * @param $order
     */
    public function receipt_page($order_or_id)
    {

        if ($order_or_id instanceof WC_Order) {
            $order = $order_or_id;
        } else {
            $order = new WC_Order($order_or_id);
        }

        if ($this->iframe == 'yes' && !$order->is_paid() && !$this->checkIfIframeShowed($order)) {
            $url = $this->generate_form($order);

            WC()->session->set('wpg_showed_' . $order->get_id(), '1');

            $this->getFrame($url);
        }
    }

    protected function getFrame($url)
    {
        if (empty($url)) {
            return;
        }

    ?>

        <iframe id="paymentFrame" src="<?php echo $url ?>" frameborder="0" style="width: 100%; height:1200px"></iframe>


        <script>
            // const paymentFrame = document.getElementById('paymentFrame');
            // paymentFrame.addEventListener('load', (ev) => {

            //     try {
            //         let link = paymentFrame.contentWindow.location.href;
            //         if (link.includes(window.location.host)) {
            //             paymentFrame.style.display = 'none';
            //             window.location.href = link;
            //         }
            //     } catch (error) {

            //     }
            // })
        </script>


<?php
    }

    protected function checkIfIframeShowed($order)
    {
        $key = 'wpg_showed_' . $order->get_id();

        if (WC()->session->get($key, false) !== false) {
            return true;
        }

        return false;
    }

    public function generate_form(WC_Order $order, bool $returnurl = true)
    {
       
        //$order = new WC_Order($);

        $order_id = $order->get_id();

        $action_adr = rtrim($this->url, '/') . $this->apimethod['checkout'];

        $customer = $this->buildCustomer($order);
        $billing_address = $this->buildBillingAddress($order);

        if ($this->settings['vat_calculation'] === 'yes') {

            $orderTotal = $order->get_total();
            $orderTax = $order->get_total_tax();

            $amountTaxFree = $orderTotal - $orderTax;
            $amount = $this->formatPrice($amountTaxFree);
        } else {
            $amount = $this->formatPrice($order->get_total());
        }

        $order_json = array(
            'number' => "$order_id",
            'description' => __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/'),
            'amount' => $amount,
            'currency' => get_woocommerce_currency(),
        );


        $str_to_hash = $order_id . $amount . get_woocommerce_currency() . __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/') . $this->password;
        $hash = sha1(md5(strtoupper($str_to_hash)));

        $mainJson = array(
            'merchant_key' => $this->secret,
            'operation'    => 'purchase',
            'order'        => $order_json,
            'customer'     => $customer,
            'billing_address' => $billing_address,
            'success_url' => $this->get_return_url($order),
            'cancel_url'   => $order->get_view_order_url(),
            'hash'         => $hash,

        );

        if (WC()->session->get('wpg_allow_tokenize') == '1') {
            $mainJson['req_token'] = true;
        }

        if (WC()->session->get('wpg_use_token') != '0' && WC()->session->get('wpg_use_token') != 'new') {
            $savedToken = WC_Payment_Tokens::get(WC()->session->get('wpg_use_token'));
            if ($savedToken->get_user_id() == get_current_user_id()) {

                $order->add_payment_token($savedToken);
                $order->save();
                $mainJson['card_token'] =  [$savedToken->get_token()];
            }
        }

        $methods = $this->method;
        if (!empty($methods)) {
            $mainJson['methods'] = $methods;
        }

      
        if (!empty($this->additional_payment_parameters)) {
            $mainJson['parameters'] = $this->buildAdditionalParameters($order_id);
        }

        if ($this->settings['vat_calculation'] === 'yes') {
            $mainJson['vat_calc'] = true;
        }

        if (!empty($this->settings['form_id'])) {
            $mainJson['form_id'] = $this->settings['form_id'];
        }

        if ($this->iframe == 'yes') {
            if ($this->url_target != 'noused') {
                $mainJson['url_target'] = '_' . $this->url_target;
            }
        }

        $response = $this->makeRequest($action_adr, $mainJson);

        if ($this->lastHttpStatus != 200) {
            throw new Exception(__( 'Unable to make payment, please contact the administrator.', 'woocommerce-gateway-wpgfull'));
        }

        if ($returnurl === true) {
            return $response['redirect_url'];
        }

        return
            '<a class="button alt" href="' . $response['redirect_url'] . '">' . __('Pay', 'woocommerce') . '</a> <a class="button cancel" href="' . $order->get_cancel_order_url() . '">' . __('Refuse payment & return to cart', 'woocommerce') . '</a>' . "\n";
    }

    public function get_transaction_url($order)
    {
        return '';
    }

    protected function buildCustomer(WC_Order $order)
    {
        $customer = array(
            'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
            'email' => $order->get_billing_email(),
        );

        return $customer;
    }

    protected function buildBillingAddress(WC_Order $order)
    {

        $stateFullname = 'NA';
        if ($order->get_billing_state()) {
            $states = WC()->countries->get_states($order->get_billing_country());
            $stateFullname = $states[$order->get_billing_state()];
        }


        $billing_address = [
            'country' => $order->get_billing_country() ? $order->get_billing_country() : 'NA',
            'state' =>  $stateFullname,
            'city' => $order->get_billing_city() ? $order->get_billing_city() : 'NA',
            'address' => $order->get_billing_address_1() ? $order->get_billing_address_1() : 'NA',
            'zip' => $order->get_billing_postcode() ? $order->get_billing_postcode() : 'NA',
            'phone' => $order->get_billing_phone() ? $order->get_billing_phone() : 'NA',
        ];



        return $billing_address;
    }

    protected function formatPrice($total)
    {
        $amount = number_format($total, 2, '.', '');
        if (in_array(get_woocommerce_currency(), $this->currencies_noexponent)) {
            $amount = number_format($total, 0, '.', '');
        } elseif (in_array(get_woocommerce_currency(), $this->currencies_3dotexponent)) {
            $amount = number_format($total, 3, '.', '');
        }
        return $amount;
    }

    protected function log($message)
    {
        if ('yes' == $this->debug) {
            $log = new WC_Logger();
            $log->add('wpgLog', $message);
        }
    }


    protected function makeRequest($url, $body)
    {
        $getter = curl_init($url); //init curl
        curl_setopt($getter, CURLOPT_POST, 1); //post
        curl_setopt($getter, CURLOPT_POSTFIELDS, json_encode($body)); //json
        curl_setopt($getter, CURLOPT_HTTPHEADER, array('Content-Type:application/json')); //header
        curl_setopt($getter, CURLOPT_RETURNTRANSFER, true);

        $result = curl_exec($getter);
        $httpcode = curl_getinfo($getter, CURLINFO_HTTP_CODE);

        $this->log('CheckoutGetUrl: ' . $url);
        $this->log('CheckoutGet: ' . json_encode($body));
        $this->log('CheckoutResponse: ' . $httpcode . ' ' . $result);

        // if ($httpcode != 200) {
        //     return false;
        // }
        $this->lastHttpStatus = $httpcode;
        $response = json_decode($result, true);

        return $response;
    }

    //gutenberg checkout
    public function modifyCheckoutFieldsNewHook($fields)
    {

        if (isset($fields['defaultFields']['phone'])) {
            $fields['data-require-phone-field']['phone']['required'] = true;
            $fields['defaultFields']['phone']['hidden'] = false;
        }

        return $fields;
    }

    public function editCheckoutContent($content)
    {

        global $post;

        if ($post->ID == wc_get_page_id('checkout')) {

            if (str_contains($content, 'data-require-phone-field="true"')) {
                return $content;
            } else {

                $content = str_replace(['data-block-name="woocommerce/checkout"', 'data-show-phone-field="false"'], ['data-block-name="woocommerce/checkout" data-require-phone-field="true"', ''], $content);
            }
        }

        return $content;
    }

    protected function buildAdditionalParameters($order_id)
    {
        $addParams = [];
        foreach ($this->additional_payment_parameters as $param) {
            $paramExploded = explode('_', $param);
            $name = $paramExploded[0];
            $class = $this->toCamelCase($param);

            $set = new $class($order_id);

            $addParams[$name] = $set->getParam(); 
        }

        return $addParams;
    }

    protected function toCamelCase($str, $capitaliseFirstChar = false)
    {
        $str = strtolower($str);
        $str = str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $str)));
        if (!$capitaliseFirstChar) {
            $str = lcfirst($str);
        }

        return $str;
    }

    public static function init_checkout_kunnect_field()
    {
        $opts = get_option('woocommerce_wpgfull_settings');
        if (
            is_array($opts) &&
            isset($opts['additional_payment_parameters']) &&
            is_array($opts['additional_payment_parameters']) &&
            in_array('kunnect', $opts['additional_payment_parameters'])
        ) {

              if (
                isset($opts['additional_payment_parameters_kunnect']) &&
                is_array($opts['additional_payment_parameters_kunnect']) &&
                !empty($opts['additional_payment_parameters_kunnect'])
            ) {

                $currencies = [];
                foreach ($opts['additional_payment_parameters_kunnect'] as $value) {
                    $currencies = ['value' => $value, 'label' => self::$cryptoCurencies[$value]];
                }

                try {
                    woocommerce_register_additional_checkout_field(
                        [
                            'id'            => 'wpg/crypto-currency',
                            'label'         => __('Crypto currency', 'woocommerce-gateway-wpgfull'),
                            'location'      => 'order',
                            'type'          => 'select', 
                            'attributes'    => array(
                                'title'            => __('Fill crypto currency', 'woocommerce-gateway-wpgfull'),
                            ),
                            'options' => $opts,
                            'required' => true,
                        ],
                    );
                } catch (\Throwable $th) {
                
                }
            }
        }
    }

    public static function modifyCheckoutFieldsOldHook($fields)
    {
        $opts = get_option('woocommerce_wpgfull_settings');
        if (
            is_array($opts) &&
            isset($opts['additional_payment_parameters']) &&
            is_array($opts['additional_payment_parameters']) &&
            in_array('kunnect', $opts['additional_payment_parameters'])
        ) {

            if (
                isset($opts['additional_payment_parameters_kunnect']) &&
                is_array($opts['additional_payment_parameters_kunnect']) &&
                !empty($opts['additional_payment_parameters_kunnect'])
            ) {

                $currencies = [];
                foreach ($opts['additional_payment_parameters_kunnect'] as $value) {
                    $currencies[$value] = self::$cryptoCurencies[$value];
                }

                $new_field = [
                    'type'         => 'select',
                    'label'        => __('Crypto currency', 'woocommerce-gateway-wpgfull'),
                    'priority'     => 1,
                    'options'      => $currencies,
                    'required'     => true,
                ];

                $fields['order']['wpg_crypto_currency'] = $new_field;
            }
        }

        return $fields;
    }

    public function saveKunnectCryptoCurrency($order, $data)
    {
        $order->add_meta_data('wpg_crypto_currency', isset( $data['wpg_crypto_currency'] ) ? $data['wpg_crypto_currency'] : '', true);
    }

    //tokenization
    public function payment_fields()
    {
        if ($this->supports('tokenization') && is_checkout()) {
            $this->tokenization_script();
            $this->saved_payment_methods();
            $this->save_payment_method_checkbox();
        }
    }

    protected function getCardTypeFromBin($bin)
    {
        $bin = preg_replace('/\D/', '', $bin); // Remove non-digits

        if (preg_match('/^4[0-9]{5,7}/', $bin)) {
            return 'visa';
        } elseif (preg_match('/^5[1-5][0-9]{4,6}/', $bin) || preg_match('/^2(2[2-9]|[3-6][0-9]|7[0-1]|720)[0-9]{2,5}/', $bin)) {
            return 'mastercard';
        } elseif (preg_match('/^3[47][0-9]{4,6}/', $bin)) {
            return 'american express';
        } elseif (preg_match('/^6(?:011|5[0-9]{2}|4[4-9][0-9])[0-9]{2,5}/', $bin)) {
            return 'discover';
        } elseif (preg_match('/^3(?:0[0-5]|[68][0-9])[0-9]{3,6}/', $bin)) {
            return 'diners';
        } elseif (preg_match('/^(?:2131|1800|35[2-8][0-9]{2})[0-9]{2,5}/', $bin)) {
            return 'jcb';
        }

        return 'unknown';
    }

    protected function allowTokenize()
    {
        if (isset($_POST["wc-{$this->id}-new-payment-method"]) && $_POST["wc-{$this->id}-new-payment-method"] == 'true') {
            WC()->session->set('wpg_allow_tokenize', '1');
            $allow_tokenize = true;
        } else {
            WC()->session->set('wpg_allow_tokenize', '0');
            $allow_tokenize = false;
        }

        if (isset($_POST["wc-{$this->id}-payment-token"])) {
            WC()->session->set('wpg_use_token', wc_clean($_POST["wc-{$this->id}-payment-token"]));
        } else {
            WC()->session->set('wpg_use_token', '0');
        }

        return $allow_tokenize;
    }

    public function get_icon()
    {

        if ($this->display_icon == 'yes') {
            $this->icon = false;
        }
        
        return parent::get_icon();
    }
}
